#!/usr/bin/env python3
"""
KW Diff Snapshot
Compare two snapshot files and report differences.

Part of the Kindware toolset.
"""

import argparse
import re
import sys
from pathlib import Path
from typing import Dict, Set, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


class SnapshotEntry:
    """Represents a file entry in a snapshot."""
    
    def __init__(self, path: str, size: int, modified: str):
        self.path = path
        self.size = size
        self.modified = modified
    
    def __eq__(self, other):
        if not isinstance(other, SnapshotEntry):
            return False
        return self.size == other.size and self.modified == other.modified
    
    def __hash__(self):
        return hash((self.path, self.size, self.modified))


def parse_snapshot(snapshot_path: Path) -> Dict[str, SnapshotEntry]:
    """
    Parse a snapshot markdown file and extract file entries.
    
    Args:
        snapshot_path: Path to the snapshot file
        
    Returns:
        Dictionary mapping file paths to SnapshotEntry objects
        
    Raises:
        RuntimeError if file cannot be parsed
    """
    try:
        with open(snapshot_path, "r", encoding="utf-8") as f:
            content = f.read()
    except Exception as e:
        raise RuntimeError(f"Failed to read snapshot file: {e}")
    
    entries = {}
    
    # Look for table rows in markdown format
    # Expected format: | path/to/file.txt | 1234 | 2025-12-23 14:30:45 |
    # The table might have headers and separator lines
    
    lines = content.split("\n")
    in_table = False
    
    for line in lines:
        line = line.strip()
        
        # Skip empty lines
        if not line:
            continue
        
        # Check if this is a table row (starts and ends with |)
        if not line.startswith("|") or not line.endswith("|"):
            continue
        
        # Split by | and clean up
        parts = [p.strip() for p in line.split("|")]
        # Remove empty first and last elements
        parts = [p for p in parts if p]
        
        # Skip header separator lines (contains dashes)
        if parts and all(set(p) <= set("-: ") for p in parts):
            in_table = True
            continue
        
        # Skip header row (contains "File", "Size", etc.)
        if parts and any(keyword in parts[0].lower() for keyword in ["file", "path"]):
            continue
        
        # Parse data rows
        if in_table and len(parts) >= 3:
            file_path = parts[0]
            size_str = parts[1]
            modified = parts[2]
            
            # Try to parse size
            try:
                # Remove any size suffixes (KB, MB, bytes, etc.)
                size_clean = re.sub(r'[^\d]', '', size_str)
                if size_clean:
                    size = int(size_clean)
                else:
                    # If we can't parse size, skip this entry
                    continue
            except ValueError:
                continue
            
            entries[file_path] = SnapshotEntry(file_path, size, modified)
    
    return entries


def compare_snapshots(
    old_entries: Dict[str, SnapshotEntry],
    new_entries: Dict[str, SnapshotEntry]
) -> Tuple[Set[str], Set[str], Set[str]]:
    """
    Compare two snapshot dictionaries.
    
    Args:
        old_entries: Entries from old snapshot
        new_entries: Entries from new snapshot
        
    Returns:
        Tuple of (added_files, removed_files, modified_files)
    """
    old_paths = set(old_entries.keys())
    new_paths = set(new_entries.keys())
    
    added = new_paths - old_paths
    removed = old_paths - new_paths
    common = old_paths & new_paths
    
    modified = set()
    for path in common:
        if old_entries[path] != new_entries[path]:
            modified.add(path)
    
    return added, removed, modified


def generate_report(
    added: Set[str],
    removed: Set[str],
    modified: Set[str]
) -> str:
    """
    Generate a markdown report of differences.
    
    Args:
        added: Set of added file paths
        removed: Set of removed file paths
        modified: Set of modified file paths
        
    Returns:
        Markdown formatted report string
    """
    lines = []
    
    lines.append("# Snapshot Comparison Report")
    lines.append("")
    
    # Check if there are any differences
    if not added and not removed and not modified:
        lines.append("No differences detected.")
        lines.append("")
        return "\n".join(lines)
    
    # Summary
    lines.append("## Summary")
    lines.append("")
    lines.append(f"- **Added**: {len(added)} file(s)")
    lines.append(f"- **Removed**: {len(removed)} file(s)")
    lines.append(f"- **Modified**: {len(modified)} file(s)")
    lines.append("")
    
    # Added files
    if added:
        lines.append("## Added")
        lines.append("")
        for path in sorted(added):
            lines.append(f"- {path}")
        lines.append("")
    
    # Removed files
    if removed:
        lines.append("## Removed")
        lines.append("")
        for path in sorted(removed):
            lines.append(f"- {path}")
        lines.append("")
    
    # Modified files
    if modified:
        lines.append("## Modified")
        lines.append("")
        for path in sorted(modified):
            lines.append(f"- {path}")
        lines.append("")
    
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(
        description="Compare two snapshot files and report differences.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s old_snapshot.md new_snapshot.md
  %(prog)s snapshot1.md snapshot2.md --output diff_report.md
        """,
    )
    
    parser.add_argument(
        "snapshot_old",
        type=str,
        help="Path to the old snapshot file",
    )
    
    parser.add_argument(
        "snapshot_new",
        type=str,
        help="Path to the new snapshot file",
    )
    
    parser.add_argument(
        "--output",
        type=str,
        metavar="FILE",
        help="Write report to a markdown file",
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Diff Snapshot {get_version()}",
    )
    
    args = parser.parse_args()
    
    # Validate snapshot files
    old_snapshot_path = Path(args.snapshot_old)
    new_snapshot_path = Path(args.snapshot_new)
    
    if not old_snapshot_path.exists():
        print(f"Error: Old snapshot file does not exist: {old_snapshot_path}", file=sys.stderr)
        return 1
    
    if not new_snapshot_path.exists():
        print(f"Error: New snapshot file does not exist: {new_snapshot_path}", file=sys.stderr)
        return 1
    
    if not old_snapshot_path.is_file():
        print(f"Error: Old snapshot path is not a file: {old_snapshot_path}", file=sys.stderr)
        return 1
    
    if not new_snapshot_path.is_file():
        print(f"Error: New snapshot path is not a file: {new_snapshot_path}", file=sys.stderr)
        return 1
    
    # Parse snapshots
    try:
        old_entries = parse_snapshot(old_snapshot_path)
    except Exception as e:
        print(f"Error: Failed to parse old snapshot: {e}", file=sys.stderr)
        return 1
    
    try:
        new_entries = parse_snapshot(new_snapshot_path)
    except Exception as e:
        print(f"Error: Failed to parse new snapshot: {e}", file=sys.stderr)
        return 1
    
    if not old_entries and not new_entries:
        print("Warning: Both snapshots are empty", file=sys.stderr)
    
    # Compare snapshots
    added, removed, modified = compare_snapshots(old_entries, new_entries)
    
    # Generate report
    report = generate_report(added, removed, modified)
    
    # Output report
    if args.output:
        try:
            output_path = Path(args.output)
            output_path.write_text(report, encoding="utf-8")
            print(f"✓ Report written to: {output_path}")
        except Exception as e:
            print(f"Error: Failed to write output file: {e}", file=sys.stderr)
            return 1
    else:
        print(report)
    
    return 0


if __name__ == "__main__":
    sys.exit(main())
