#!/usr/bin/env python3
"""
KW File Indexer
A lightweight utility that scans a directory and generates a structured file index.
"""

import argparse
import os
import sys
from pathlib import Path
from typing import Dict, List


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


INDEX_FILENAME = "FILE_INDEX.md"


def parse_arguments():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser(
        description="Generate a structured file index for a directory."
    )
    parser.add_argument(
        "path",
        type=str,
        help="Path to the directory to scan"
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print progress while scanning"
    )
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW File Indexer {get_version()}"
    )
    return parser.parse_args()


def scan_directory(root_path: Path, verbose: bool = False) -> Dict[str, List[Path]]:
    """
    Recursively scan a directory and organize files by folder.
    
    Args:
        root_path: Root directory to scan
        verbose: Whether to print progress
    
    Returns:
        Dictionary mapping relative folder paths to lists of files
    """
    files_by_folder: Dict[str, List[Path]] = {}
    
    try:
        for dirpath, dirnames, filenames in os.walk(root_path):
            dir_path = Path(dirpath)
            
            # Calculate relative path from root
            try:
                rel_dir = dir_path.relative_to(root_path)
            except ValueError:
                rel_dir = Path(".")
            
            # Convert to string, use "." for root
            rel_dir_str = "." if str(rel_dir) == "." else str(rel_dir)
            
            # Filter out the index file itself
            valid_files = [
                f for f in filenames 
                if f != INDEX_FILENAME
            ]
            
            if valid_files:
                if verbose:
                    print(f"Scanning: {rel_dir_str}")
                
                # Store file paths
                files_by_folder[rel_dir_str] = [
                    Path(f) for f in sorted(valid_files, key=str.lower)
                ]
    
    except PermissionError as e:
        print(f"Warning: Permission denied for some files/folders: {e}", file=sys.stderr)
    
    return files_by_folder


def generate_index(files_by_folder: Dict[str, List[Path]], output_path: Path, verbose: bool = False):
    """
    Generate the FILE_INDEX.md file.
    
    Args:
        files_by_folder: Dictionary mapping folder paths to file lists
        output_path: Path where FILE_INDEX.md will be written
        verbose: Whether to print progress
    """
    if verbose:
        print(f"\nGenerating index: {output_path}")
    
    with open(output_path, "w", encoding="utf-8") as f:
        f.write("# File Index\n\n")
        
        # Sort folders for consistent output
        sorted_folders = sorted(files_by_folder.keys())
        
        for folder in sorted_folders:
            files = files_by_folder[folder]
            
            if not files:
                continue
            
            # Write folder heading
            f.write(f"## {folder}\n\n")
            
            # Write file list
            for file_path in files:
                file_name = file_path.name
                file_ext = file_path.suffix if file_path.suffix else "(no extension)"
                f.write(f"- {file_name} - {file_ext}\n")
            
            f.write("\n")
    
    if verbose:
        print(f"Index generated successfully: {output_path}")


def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Validate target path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if args.verbose:
        print(f"Scanning directory: {target_path}\n")
    
    # Scan the directory
    try:
        files_by_folder = scan_directory(target_path, verbose=args.verbose)
    except Exception as e:
        print(f"Error during scan: {e}", file=sys.stderr)
        sys.exit(1)
    
    # Generate the index file
    index_path = target_path / INDEX_FILENAME
    try:
        generate_index(files_by_folder, index_path, verbose=args.verbose)
    except PermissionError:
        print(f"Error: Permission denied writing to: {index_path}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error generating index: {e}", file=sys.stderr)
        sys.exit(1)
    
    if not args.verbose:
        print(f"Index generated: {index_path}")


if __name__ == "__main__":
    main()
