#!/usr/bin/env python3
"""
KW Folder Snapshot
A lightweight utility that captures timestamped snapshots of directory structure and file metadata.
"""

import argparse
import os
import sys
from datetime import datetime
from pathlib import Path
from typing import List, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


def parse_arguments():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser(
        description="Capture a timestamped snapshot of a directory's structure and file metadata."
    )
    parser.add_argument(
        "path",
        type=str,
        help="Path to the directory to snapshot"
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print progress while scanning"
    )
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Folder Snapshot {get_version()}"
    )
    return parser.parse_args()


def get_snapshot_filename() -> str:
    """Generate snapshot filename with current timestamp."""
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    return f"SNAPSHOT_{timestamp}.md"


def scan_directory(root_path: Path, snapshot_filename: str, verbose: bool = False) -> List[Tuple[str, int, str]]:
    """
    Recursively scan a directory and collect file metadata.
    
    Args:
        root_path: Root directory to scan
        snapshot_filename: Name of the snapshot file to exclude
        verbose: Whether to print progress
    
    Returns:
        List of tuples containing (relative_path, size_bytes, modified_timestamp)
    """
    file_data = []
    
    try:
        for dirpath, dirnames, filenames in os.walk(root_path):
            dir_path = Path(dirpath)
            
            for filename in filenames:
                # Skip the snapshot file itself
                if filename.startswith("SNAPSHOT_") and filename.endswith(".md"):
                    continue
                
                file_path = dir_path / filename
                
                try:
                    # Get file stats
                    stats = file_path.stat()
                    
                    # Calculate relative path
                    try:
                        rel_path = file_path.relative_to(root_path)
                    except ValueError:
                        rel_path = file_path
                    
                    # Get file size and modification time
                    size_bytes = stats.st_size
                    modified_timestamp = datetime.fromtimestamp(stats.st_mtime).strftime("%Y-%m-%d %H:%M:%S")
                    
                    file_data.append((str(rel_path), size_bytes, modified_timestamp))
                    
                    if verbose:
                        print(f"Scanned: {rel_path}")
                
                except PermissionError:
                    print(f"Warning: Permission denied - {file_path}", file=sys.stderr)
                except Exception as e:
                    print(f"Warning: Error reading {file_path} - {e}", file=sys.stderr)
    
    except PermissionError as e:
        print(f"Warning: Permission denied for some directories: {e}", file=sys.stderr)
    
    # Sort by relative path for consistent output
    file_data.sort(key=lambda x: x[0].lower())
    
    return file_data


def generate_snapshot(file_data: List[Tuple[str, int, str]], output_path: Path, root_path: Path, verbose: bool = False):
    """
    Generate the snapshot markdown file.
    
    Args:
        file_data: List of tuples containing file metadata
        output_path: Path where snapshot file will be written
        root_path: Root path that was scanned
        verbose: Whether to print progress
    """
    if verbose:
        print(f"\nGenerating snapshot: {output_path}")
    
    snapshot_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    with open(output_path, "w", encoding="utf-8") as f:
        f.write("# Folder Snapshot\n\n")
        f.write(f"**Snapshot Time:** {snapshot_time}\n\n")
        f.write(f"**Root Path:** {root_path}\n\n")
        f.write("---\n\n")
        
        if not file_data:
            f.write("*No files found in directory.*\n")
        else:
            f.write("## Files\n\n")
            
            for rel_path, size_bytes, modified_time in file_data:
                f.write(f"### {rel_path}\n\n")
                f.write(f"- **Size:** {size_bytes} bytes\n")
                f.write(f"- **Last Modified:** {modified_time}\n\n")
    
    if verbose:
        print(f"Snapshot generated successfully: {output_path}")
        print(f"Total files: {len(file_data)}")


def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Validate target path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if args.verbose:
        print(f"Scanning directory: {target_path}\n")
    
    # Generate snapshot filename
    snapshot_filename = get_snapshot_filename()
    
    # Scan the directory
    try:
        file_data = scan_directory(target_path, snapshot_filename, verbose=args.verbose)
    except Exception as e:
        print(f"Error during scan: {e}", file=sys.stderr)
        sys.exit(1)
    
    # Generate the snapshot file
    snapshot_path = target_path / snapshot_filename
    try:
        generate_snapshot(file_data, snapshot_path, target_path, verbose=args.verbose)
    except PermissionError:
        print(f"Error: Permission denied writing to: {snapshot_path}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error generating snapshot: {e}", file=sys.stderr)
        sys.exit(1)
    
    if not args.verbose:
        print(f"Snapshot created: {snapshot_path}")


if __name__ == "__main__":
    main()
