#!/usr/bin/env python3
"""
KW Hash Generator
Generate cryptographic hashes for files to verify integrity.

Part of the Kindware toolset.
"""

import argparse
import hashlib
import sys
from pathlib import Path
from typing import List, TextIO


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


# Buffer size for reading files (1MB chunks)
BUFFER_SIZE = 1024 * 1024


def calculate_hash(file_path: Path, algorithm: str) -> str:
    """
    Calculate the hash of a file using the specified algorithm.
    
    Args:
        file_path: Path to the file to hash
        algorithm: Hash algorithm name (md5, sha1, sha256)
        
    Returns:
        Hexadecimal hash string
        
    Raises:
        Exception if file cannot be read
    """
    if algorithm == "md5":
        hasher = hashlib.md5()
    elif algorithm == "sha1":
        hasher = hashlib.sha1()
    elif algorithm == "sha256":
        hasher = hashlib.sha256()
    else:
        raise ValueError(f"Unsupported algorithm: {algorithm}")
    
    with open(file_path, "rb") as f:
        while True:
            chunk = f.read(BUFFER_SIZE)
            if not chunk:
                break
            hasher.update(chunk)
    
    return hasher.hexdigest()


def process_file(file_path: Path, algorithms: List[str], output_handle: TextIO, verbose: bool) -> int:
    """
    Process a single file and generate hashes for specified algorithms.
    
    Args:
        file_path: Path to the file to process
        algorithms: List of algorithm names to use
        output_handle: File handle to write output to
        verbose: Whether to print progress messages
        
    Returns:
        Number of successful hashes generated
    """
    success_count = 0
    filename = file_path.name
    
    for algorithm in algorithms:
        try:
            if verbose:
                print(f"Hashing {filename} with {algorithm.upper()}...", file=sys.stderr)
            
            hash_value = calculate_hash(file_path, algorithm)
            output_handle.write(f"{filename} | {algorithm.upper()} | {hash_value}\n")
            success_count += 1
            
        except Exception as e:
            print(f"Error: Failed to hash {filename} with {algorithm.upper()}: {e}", file=sys.stderr)
    
    return success_count


def main():
    parser = argparse.ArgumentParser(
        description="Generate cryptographic hashes for files.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s file.txt --sha256
  %(prog)s /path/to/file --md5 --sha1
  %(prog)s /path/to/directory --all
  %(prog)s file.txt --sha256 --output hashes.txt
  %(prog)s . --all --verbose
        """,
    )
    
    parser.add_argument(
        "path",
        type=str,
        help="File or directory to hash",
    )
    
    parser.add_argument(
        "--md5",
        action="store_true",
        help="Generate MD5 hash",
    )
    
    parser.add_argument(
        "--sha1",
        action="store_true",
        help="Generate SHA1 hash",
    )
    
    parser.add_argument(
        "--sha256",
        action="store_true",
        help="Generate SHA256 hash",
    )
    
    parser.add_argument(
        "--all",
        action="store_true",
        help="Generate all supported hashes (MD5, SHA1, SHA256)",
    )
    
    parser.add_argument(
        "--output",
        type=str,
        metavar="FILE",
        help="Write results to a file instead of stdout",
    )
    
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print progress while hashing",
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Hash Generator {get_version()}",
    )
    
    args = parser.parse_args()
    
    # Validate path
    target_path = Path(args.path)
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        return 1
    
    # Don't follow symlinks
    if target_path.is_symlink():
        print(f"Error: Path is a symlink (not supported): {target_path}", file=sys.stderr)
        return 1
    
    # Determine which algorithms to use
    algorithms = []
    
    if args.all:
        algorithms = ["md5", "sha1", "sha256"]
    else:
        if args.md5:
            algorithms.append("md5")
        if args.sha1:
            algorithms.append("sha1")
        if args.sha256:
            algorithms.append("sha256")
    
    if not algorithms:
        print("Error: No hash algorithm specified. Use --md5, --sha1, --sha256, or --all", file=sys.stderr)
        parser.print_help(sys.stderr)
        return 1
    
    # Collect files to process
    files_to_process = []
    
    if target_path.is_file():
        files_to_process.append(target_path)
    elif target_path.is_dir():
        # Non-recursive directory scan
        try:
            for item in target_path.iterdir():
                # Skip symlinks
                if item.is_symlink():
                    if args.verbose:
                        print(f"Skipping symlink: {item.name}", file=sys.stderr)
                    continue
                
                # Only process files
                if item.is_file():
                    files_to_process.append(item)
        except Exception as e:
            print(f"Error: Failed to read directory: {e}", file=sys.stderr)
            return 1
    else:
        print(f"Error: Path is neither a file nor a directory: {target_path}", file=sys.stderr)
        return 1
    
    if not files_to_process:
        print("Warning: No files to process", file=sys.stderr)
        return 0
    
    # Open output handle
    output_handle = sys.stdout
    should_close = False
    
    if args.output:
        try:
            output_handle = open(args.output, "w", encoding="utf-8")
            should_close = True
            if args.verbose:
                print(f"Writing output to: {args.output}", file=sys.stderr)
        except Exception as e:
            print(f"Error: Failed to open output file: {e}", file=sys.stderr)
            return 1
    
    # Process files
    total_hashes = 0
    processed_files = 0
    
    try:
        if args.verbose:
            print(f"Processing {len(files_to_process)} file(s) with {len(algorithms)} algorithm(s)...", file=sys.stderr)
            print()
        
        for file_path in files_to_process:
            hashes_generated = process_file(file_path, algorithms, output_handle, args.verbose)
            if hashes_generated > 0:
                processed_files += 1
                total_hashes += hashes_generated
        
        if args.verbose:
            print(f"\n✓ Processed {processed_files}/{len(files_to_process)} file(s)", file=sys.stderr)
            print(f"✓ Generated {total_hashes} hash(es)", file=sys.stderr)
        
        return 0
    
    finally:
        if should_close:
            output_handle.close()


if __name__ == "__main__":
    sys.exit(main())
