#!/usr/bin/env python3
"""
KW Batch Renamer
A lightweight utility for safely batch renaming files in a directory.
"""

import argparse
import os
import sys
from pathlib import Path
from typing import List, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


def parse_arguments():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser(
        description="Batch rename files in a directory safely and predictably."
    )
    parser.add_argument(
        "path",
        type=str,
        help="Directory containing files to rename"
    )
    parser.add_argument(
        "--prefix",
        type=str,
        default="",
        help="Text added before filename"
    )
    parser.add_argument(
        "--suffix",
        type=str,
        default="",
        help="Text added after filename (before extension)"
    )
    parser.add_argument(
        "--start-number",
        type=int,
        default=1,
        help="Starting number for sequential numbering (default: 1)"
    )
    parser.add_argument(
        "--padding",
        type=int,
        default=0,
        help="Zero padding width for numbers (e.g., 3 -> 001)"
    )
    parser.add_argument(
        "--lower",
        action="store_true",
        help="Convert filename to lowercase"
    )
    parser.add_argument(
        "--upper",
        action="store_true",
        help="Convert filename to uppercase"
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Show what would happen without renaming files"
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print detailed output"
    )
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Batch Renamer {get_version()}"
    )
    return parser.parse_args()


def get_files(directory: Path, verbose: bool = False) -> List[Path]:
    """
    Get all files in directory (non-recursive, skip subdirectories).
    
    Args:
        directory: Directory to scan
        verbose: Whether to print progress
    
    Returns:
        List of file paths
    """
    files = []
    
    try:
        for item in directory.iterdir():
            if item.is_file():
                files.append(item)
            elif verbose and item.is_dir():
                print(f"Skipping directory: {item.name}")
    except PermissionError as e:
        print(f"Error: Permission denied accessing directory: {e}", file=sys.stderr)
        sys.exit(1)
    
    # Sort files for consistent processing
    files.sort(key=lambda x: x.name.lower())
    
    return files


def generate_new_name(
    original_path: Path,
    prefix: str,
    suffix: str,
    number: int,
    padding: int,
    lower: bool,
    upper: bool
) -> str:
    """
    Generate new filename based on options.
    
    Processing order:
    1. Case transformation (--lower / --upper)
    2. Prefix
    3. Base filename
    4. Optional numbering
    5. Suffix
    6. Extension (preserved)
    
    Args:
        original_path: Original file path
        prefix: Text to add before filename
        suffix: Text to add after filename (before extension)
        number: Sequential number (0 if not used)
        padding: Zero padding width
        lower: Convert to lowercase
        upper: Convert to uppercase
    
    Returns:
        New filename
    """
    # Get base name and extension
    stem = original_path.stem
    ext = original_path.suffix
    
    # Step 1: Case transformation
    if lower:
        stem = stem.lower()
    elif upper:
        stem = stem.upper()
    
    # Step 2-5: Build new name
    new_name_parts = []
    
    # Prefix
    if prefix:
        new_name_parts.append(prefix)
    
    # Base filename
    new_name_parts.append(stem)
    
    # Numbering
    if padding > 0:
        new_name_parts.append(str(number).zfill(padding))
    elif number > 0:
        new_name_parts.append(str(number))
    
    # Suffix
    if suffix:
        new_name_parts.append(suffix)
    
    # Combine parts
    new_stem = "".join(new_name_parts)
    
    # Step 6: Add extension back
    return new_stem + ext


def rename_files(
    files: List[Path],
    prefix: str,
    suffix: str,
    start_number: int,
    padding: int,
    lower: bool,
    upper: bool,
    dry_run: bool,
    verbose: bool
) -> Tuple[int, int]:
    """
    Rename files according to options.
    
    Args:
        files: List of file paths to rename
        prefix: Prefix to add
        suffix: Suffix to add
        start_number: Starting number
        padding: Zero padding width
        lower: Convert to lowercase
        upper: Convert to uppercase
        dry_run: Don't actually rename
        verbose: Print detailed output
    
    Returns:
        Tuple of (successful_count, skipped_count)
    """
    successful = 0
    skipped = 0
    
    # Determine if we should use numbering
    use_numbering = padding > 0 or (prefix or suffix or lower or upper)
    
    for idx, file_path in enumerate(files):
        # Calculate number for this file
        current_number = start_number + idx if use_numbering else 0
        
        # Generate new filename
        new_filename = generate_new_name(
            file_path,
            prefix,
            suffix,
            current_number if use_numbering else 0,
            padding,
            lower,
            upper
        )
        
        new_path = file_path.parent / new_filename
        
        # Check if file would be unchanged
        if file_path.name == new_filename:
            if verbose:
                print(f"Unchanged: {file_path.name}")
            skipped += 1
            continue
        
        # Check if target already exists
        if new_path.exists():
            print(f"Skipping (target exists): {file_path.name} -> {new_filename}", file=sys.stderr)
            skipped += 1
            continue
        
        # Perform rename or dry-run
        if dry_run:
            print(f"Would rename: {file_path.name} -> {new_filename}")
            successful += 1
        else:
            try:
                file_path.rename(new_path)
                if verbose:
                    print(f"Renamed: {file_path.name} -> {new_filename}")
                successful += 1
            except Exception as e:
                print(f"Error renaming {file_path.name}: {e}", file=sys.stderr)
                skipped += 1
    
    return successful, skipped


def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Validate target path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    # Check for conflicting options
    if args.lower and args.upper:
        print("Error: Cannot use both --lower and --upper", file=sys.stderr)
        sys.exit(1)
    
    if args.verbose:
        print(f"Processing directory: {target_path}")
        if args.dry_run:
            print("DRY RUN MODE - No files will be renamed\n")
    
    # Get files to process
    files = get_files(target_path, verbose=args.verbose)
    
    if not files:
        print("No files found to rename.")
        return
    
    if args.verbose:
        print(f"\nFound {len(files)} file(s) to process\n")
    
    # Rename files
    successful, skipped = rename_files(
        files,
        args.prefix,
        args.suffix,
        args.start_number,
        args.padding,
        args.lower,
        args.upper,
        args.dry_run,
        args.verbose
    )
    
    # Print summary
    if args.dry_run:
        print(f"\nDry run complete: {successful} file(s) would be renamed, {skipped} skipped")
    else:
        print(f"\nRename complete: {successful} file(s) renamed, {skipped} skipped")


if __name__ == "__main__":
    main()
