#!/usr/bin/env python3
"""
KW Folder Organizer - Organize files into category folders by file type
Version: 0.1.0

A cross-platform command-line utility that automatically organizes files in a
directory into predefined category folders based on file extensions.
"""

import argparse
import sys
from pathlib import Path
from typing import Dict, Set, Tuple, List
import shutil


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


# Exit codes
EXIT_SUCCESS = 0
EXIT_USER_ERROR = 1
EXIT_UNEXPECTED_ERROR = 2


# File extension categories (case-insensitive)
CATEGORIES: Dict[str, Set[str]] = {
    "Images": {".jpg", ".jpeg", ".png", ".gif", ".bmp", ".webp"},
    "Videos": {".mp4", ".mkv", ".avi", ".mov", ".wmv"},
    "Audio": {".mp3", ".wav", ".flac", ".aac", ".ogg"},
    "Documents": {".pdf", ".doc", ".docx", ".txt", ".md", ".rtf"},
    "Archives": {".zip", ".rar", ".7z", ".tar", ".gz"},
    "Code": {".py", ".js", ".ts", ".cs", ".java", ".cpp", ".h", ".json", ".php", ".html", ".css", ".ps1", ".sh"},
}

# Default category for unmatched extensions
DEFAULT_CATEGORY = "Other"


def get_category(file_path: Path) -> str:
    """
    Determine the category for a file based on its extension.
    
    Args:
        file_path: Path to the file
        
    Returns:
        Category name (e.g., "Images", "Videos", "Other")
    """
    # Get lowercase extension for case-insensitive matching
    ext = file_path.suffix.lower()
    
    # Search through categories for matching extension
    for category, extensions in CATEGORIES.items():
        if ext in extensions:
            return category
    
    # Default category if no match found
    return DEFAULT_CATEGORY


def resolve_collision(target_path: Path) -> Path:
    """
    Resolve filename collisions by appending _1, _2, etc. to the filename.
    
    Args:
        target_path: The desired target path
        
    Returns:
        A unique path that doesn't conflict with existing files
    """
    if not target_path.exists():
        return target_path
    
    # Extract stem (filename without extension) and suffix
    stem = target_path.stem
    suffix = target_path.suffix
    parent = target_path.parent
    
    # Try incrementing numbers until we find an available name
    counter = 1
    while True:
        new_name = f"{stem}_{counter}{suffix}"
        new_path = parent / new_name
        if not new_path.exists():
            return new_path
        counter += 1


def organize_files(
    target_dir: Path,
    dry_run: bool = False,
    verbose: bool = False
) -> Tuple[int, int]:
    """
    Organize files in the target directory into category folders.
    
    Args:
        target_dir: Directory to organize
        dry_run: If True, don't actually move files
        verbose: If True, print detailed information
        
    Returns:
        Tuple of (files_moved, categories_created)
    """
    files_moved = 0
    categories_created_set: Set[str] = set()
    
    # Get all files in the top level of the directory (non-recursive)
    try:
        all_items = list(target_dir.iterdir())
    except PermissionError as e:
        print(f"Error: Permission denied accessing directory: {e}", file=sys.stderr)
        sys.exit(EXIT_USER_ERROR)
    
    # Filter to only files (exclude directories)
    files = [item for item in all_items if item.is_file()]
    
    if not files:
        if verbose:
            print("No files found to organize.")
        return 0, 0
    
    # Process each file
    for file_path in files:
        # Determine the category
        category = get_category(file_path)
        category_folder = target_dir / category
        
        # Skip if file is already in a category folder
        # (We only process top-level files, so this shouldn't happen)
        if file_path.parent != target_dir:
            continue
        
        # Create category folder if it doesn't exist
        if not category_folder.exists():
            if not dry_run:
                try:
                    category_folder.mkdir(parents=True, exist_ok=True)
                    categories_created_set.add(category)
                except OSError as e:
                    print(
                        f"Error: Failed to create folder '{category}': {e}",
                        file=sys.stderr
                    )
                    continue
            else:
                categories_created_set.add(category)
        
        # Determine target path
        target_path = category_folder / file_path.name
        
        # Check if file already exists in target location with same name
        if target_path.exists():
            # Resolve collision
            target_path = resolve_collision(target_path)
        
        # Move the file (or simulate in dry-run mode)
        try:
            if dry_run:
                # Dry run mode: just report what would happen
                relative_target = target_path.relative_to(target_dir)
                print(f"[DRY-RUN] Would move: {file_path.name} -> {relative_target}")
            else:
                # Actually move the file
                shutil.move(str(file_path), str(target_path))
                files_moved += 1
                
                if verbose:
                    relative_target = target_path.relative_to(target_dir)
                    print(f"Moved: {file_path.name} -> {relative_target}")
        except (OSError, shutil.Error) as e:
            print(
                f"Error: Failed to move '{file_path.name}': {e}",
                file=sys.stderr
            )
            continue
    
    return files_moved, len(categories_created_set)


def main() -> int:
    """
    Main entry point for the KW Folder Organizer.
    
    Returns:
        Exit code (0 for success, 1 for user error, 2 for unexpected error)
    """
    # Set up argument parser
    parser = argparse.ArgumentParser(
        prog="kw-folder-organizer",
        description="Organize files into category folders by file type.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Categories:
  Images:    .jpg, .jpeg, .png, .gif, .bmp, .webp
  Videos:    .mp4, .mkv, .avi, .mov, .wmv
  Audio:     .mp3, .wav, .flac, .aac, .ogg
  Documents: .pdf, .doc, .docx, .txt, .md, .rtf
  Archives:  .zip, .rar, .7z, .tar, .gz
  Code:      .py, .js, .ts, .cs, .java, .cpp, .h
  Other:     Everything else

Examples:
  kw-folder-organizer /path/to/folder
  kw-folder-organizer . --dry-run
  kw-folder-organizer ~/Downloads --verbose
        """
    )
    
    # Positional argument: target directory
    parser.add_argument(
        "path",
        type=str,
        help="Path to the directory to organize"
    )
    
    # Optional flags
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Show what would happen without moving files"
    )
    
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print detailed operations"
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Folder Organizer {get_version()}"
    )
    
    # Parse arguments
    args = parser.parse_args()
    
    # Convert path string to Path object
    target_path = Path(args.path).resolve()
    
    # Validate the target path
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        return EXIT_USER_ERROR
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        return EXIT_USER_ERROR
    
    # Check if we have permission to access the directory
    if not target_path.exists():
        print(f"Error: Cannot access directory: {target_path}", file=sys.stderr)
        return EXIT_USER_ERROR
    
    try:
        # Perform the organization
        files_moved, categories_created = organize_files(
            target_path,
            dry_run=args.dry_run,
            verbose=args.verbose
        )
        
        # Print summary (unless verbose already printed details)
        if not args.verbose and not args.dry_run:
            if files_moved > 0:
                print(f"Organized {files_moved} file(s) into {categories_created} category/categories.")
            else:
                print("No files to organize.")
        elif args.dry_run and not args.verbose:
            # In dry-run mode, also show a summary
            if files_moved == 0 and categories_created == 0:
                print("[DRY-RUN] No files to organize.")
        
        return EXIT_SUCCESS
        
    except KeyboardInterrupt:
        print("\nOperation cancelled by user.", file=sys.stderr)
        return EXIT_USER_ERROR
    except Exception as e:
        print(f"Unexpected error: {e}", file=sys.stderr)
        return EXIT_UNEXPECTED_ERROR


if __name__ == "__main__":
    sys.exit(main())
