#!/usr/bin/env python3
"""
KW Project Cleaner
A lightweight utility that removes common build artifacts, cache folders, and junk files.
"""

import argparse
import os
import shutil
import sys
from pathlib import Path
from typing import List, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


# Predefined targets to clean
TARGET_FOLDERS = {
    "__pycache__",
    "node_modules",
    "bin",
    "obj",
    ".vs",
    ".idea",
    "dist",
    "build"
}

TARGET_FILES = {
    ".DS_Store",
    "Thumbs.db"
}


def parse_arguments():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser(
        description="Safely remove build artifacts, cache folders, and junk files from projects."
    )
    parser.add_argument(
        "path",
        type=str,
        help="Directory to clean"
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Show what would be deleted without removing anything"
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print detailed operations"
    )
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Project Cleaner {get_version()}"
    )
    return parser.parse_args()


def scan_directory(root_path: Path, verbose: bool = False) -> Tuple[List[Path], List[Path]]:
    """
    Recursively scan directory for target folders and files.
    
    Args:
        root_path: Root directory to scan
        verbose: Whether to print progress
    
    Returns:
        Tuple of (folders_to_delete, files_to_delete)
    """
    folders_to_delete = []
    files_to_delete = []
    
    if verbose:
        print(f"Scanning: {root_path}\n")
    
    try:
        for dirpath, dirnames, filenames in os.walk(root_path, followlinks=False):
            dir_path = Path(dirpath)
            
            # Check for target folders
            for dirname in list(dirnames):
                if dirname in TARGET_FOLDERS:
                    folder_path = dir_path / dirname
                    folders_to_delete.append(folder_path)
                    
                    # Remove from dirnames to prevent walking into it
                    dirnames.remove(dirname)
                    
                    if verbose:
                        print(f"Found folder: {folder_path}")
            
            # Check for target files
            for filename in filenames:
                if filename in TARGET_FILES:
                    file_path = dir_path / filename
                    files_to_delete.append(file_path)
                    
                    if verbose:
                        print(f"Found file: {file_path}")
    
    except PermissionError as e:
        print(f"Warning: Permission denied for some directories: {e}", file=sys.stderr)
    
    return folders_to_delete, files_to_delete


def delete_folders(folders: List[Path], dry_run: bool, verbose: bool) -> Tuple[int, int]:
    """
    Delete target folders.
    
    Args:
        folders: List of folder paths to delete
        dry_run: Whether this is a dry run
        verbose: Whether to print detailed output
    
    Returns:
        Tuple of (successful_count, failed_count)
    """
    successful = 0
    failed = 0
    
    for folder_path in folders:
        prefix = "[DRY-RUN] " if dry_run else ""
        
        try:
            if dry_run:
                if verbose:
                    print(f"{prefix}Would delete folder: {folder_path}")
                successful += 1
            else:
                shutil.rmtree(folder_path)
                if verbose:
                    print(f"Deleted folder: {folder_path}")
                successful += 1
        except PermissionError:
            print(f"{prefix}Error: Permission denied - {folder_path}", file=sys.stderr)
            failed += 1
        except Exception as e:
            print(f"{prefix}Error deleting {folder_path}: {e}", file=sys.stderr)
            failed += 1
    
    return successful, failed


def delete_files(files: List[Path], dry_run: bool, verbose: bool) -> Tuple[int, int]:
    """
    Delete target files.
    
    Args:
        files: List of file paths to delete
        dry_run: Whether this is a dry run
        verbose: Whether to print detailed output
    
    Returns:
        Tuple of (successful_count, failed_count)
    """
    successful = 0
    failed = 0
    
    for file_path in files:
        prefix = "[DRY-RUN] " if dry_run else ""
        
        try:
            if dry_run:
                if verbose:
                    print(f"{prefix}Would delete file: {file_path}")
                successful += 1
            else:
                file_path.unlink()
                if verbose:
                    print(f"Deleted file: {file_path}")
                successful += 1
        except PermissionError:
            print(f"{prefix}Error: Permission denied - {file_path}", file=sys.stderr)
            failed += 1
        except Exception as e:
            print(f"{prefix}Error deleting {file_path}: {e}", file=sys.stderr)
            failed += 1
    
    return successful, failed


def get_size_summary(folders: List[Path], files: List[Path]) -> str:
    """
    Calculate approximate size of targets.
    
    Args:
        folders: List of folders to check
        files: List of files to check
    
    Returns:
        Human-readable size string
    """
    total_size = 0
    
    # Calculate folder sizes
    for folder in folders:
        try:
            for dirpath, dirnames, filenames in os.walk(folder, followlinks=False):
                for filename in filenames:
                    try:
                        file_path = Path(dirpath) / filename
                        total_size += file_path.stat().st_size
                    except:
                        pass
        except:
            pass
    
    # Calculate file sizes
    for file_path in files:
        try:
            total_size += file_path.stat().st_size
        except:
            pass
    
    # Convert to human-readable format
    if total_size < 1024:
        return f"{total_size} bytes"
    elif total_size < 1024 * 1024:
        return f"{total_size / 1024:.2f} KB"
    elif total_size < 1024 * 1024 * 1024:
        return f"{total_size / (1024 * 1024):.2f} MB"
    else:
        return f"{total_size / (1024 * 1024 * 1024):.2f} GB"


def main():
    """Main entry point."""
    args = parse_arguments()
    
    # Validate target path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        sys.exit(1)
    
    if args.dry_run:
        print("=== DRY RUN MODE - No files will be deleted ===\n")
    
    if args.verbose:
        print(f"Cleaning directory: {target_path}")
        print(f"Target folders: {', '.join(sorted(TARGET_FOLDERS))}")
        print(f"Target files: {', '.join(sorted(TARGET_FILES))}\n")
    
    # Scan for targets
    folders, files = scan_directory(target_path, verbose=args.verbose)
    
    if not folders and not files:
        print("No targets found to clean.")
        return
    
    # Show summary
    print(f"\nFound {len(folders)} folder(s) and {len(files)} file(s) to clean")
    
    if args.verbose:
        size_summary = get_size_summary(folders, files)
        print(f"Approximate size: {size_summary}")
    
    print()
    
    # Delete folders
    if folders:
        folder_success, folder_fail = delete_folders(folders, args.dry_run, args.verbose)
        
        if not args.verbose:
            prefix = "[DRY-RUN] " if args.dry_run else ""
            if args.dry_run:
                print(f"{prefix}Would delete {folder_success} folder(s)")
            else:
                print(f"Deleted {folder_success} folder(s)")
            if folder_fail > 0:
                print(f"Failed to delete {folder_fail} folder(s)")
    
    # Delete files
    if files:
        file_success, file_fail = delete_files(files, args.dry_run, args.verbose)
        
        if not args.verbose:
            prefix = "[DRY-RUN] " if args.dry_run else ""
            if args.dry_run:
                print(f"{prefix}Would delete {file_success} file(s)")
            else:
                print(f"Deleted {file_success} file(s)")
            if file_fail > 0:
                print(f"Failed to delete {file_fail} file(s)")
    
    # Final message
    if args.dry_run:
        print("\n=== Dry run complete - No changes made ===")
    else:
        print("\n=== Cleanup complete ===")


if __name__ == "__main__":
    main()
