#!/usr/bin/env python3
"""
KW Command Logger
Run a command and capture its output to a timestamped log file.

Part of the Kindware toolset.
"""

import argparse
import subprocess
import sys
from datetime import datetime
from pathlib import Path
from typing import Optional, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


def get_log_filename() -> str:
    """
    Generate a timestamped log filename.
    
    Returns:
        Filename in format COMMAND_LOG_YYYY-MM-DD_HH-MM-SS.txt
    """
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    return f"COMMAND_LOG_{timestamp}.txt"


def run_command(
    command: list,
    cwd: Optional[Path],
    timeout: Optional[int],
    echo: bool
) -> Tuple[int, str, str]:
    """
    Run a command and capture its output.
    
    Args:
        command: Command and arguments as list
        cwd: Working directory to run command in
        timeout: Optional timeout in seconds
        echo: Whether to echo output to console
        
    Returns:
        Tuple of (exit_code, stdout, stderr)
    """
    stdout_lines = []
    stderr_lines = []
    
    try:
        # Run the process
        process = subprocess.Popen(
            command,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            cwd=cwd,
            text=True,
            bufsize=1,  # Line buffered
        )
        
        # Read output in real-time
        try:
            stdout, stderr = process.communicate(timeout=timeout)
            
            stdout_lines = stdout.splitlines(keepends=True) if stdout else []
            stderr_lines = stderr.splitlines(keepends=True) if stderr else []
            
            if echo:
                if stdout:
                    print(stdout, end="")
                if stderr:
                    print(stderr, end="", file=sys.stderr)
            
            exit_code = process.returncode
            
        except subprocess.TimeoutExpired:
            process.kill()
            stdout, stderr = process.communicate()
            
            stdout_lines = stdout.splitlines(keepends=True) if stdout else []
            stderr_lines = stderr.splitlines(keepends=True) if stderr else []
            
            if echo:
                if stdout:
                    print(stdout, end="")
                if stderr:
                    print(stderr, end="", file=sys.stderr)
            
            stderr_lines.append(f"\n[TIMEOUT] Command exceeded {timeout} seconds\n")
            exit_code = 124  # Standard timeout exit code
    
    except FileNotFoundError:
        stderr_lines.append(f"[ERROR] Command not found: {command[0]}\n")
        exit_code = 127
    except Exception as e:
        stderr_lines.append(f"[ERROR] Failed to execute command: {e}\n")
        exit_code = 1
    
    stdout_text = "".join(stdout_lines)
    stderr_text = "".join(stderr_lines)
    
    return exit_code, stdout_text, stderr_text


def write_log_file(
    log_path: Path,
    command: list,
    cwd: Path,
    start_time: datetime,
    end_time: datetime,
    exit_code: int,
    stdout: str,
    stderr: str
) -> None:
    """
    Write the log file with command details and output.
    
    Args:
        log_path: Path to write the log file
        command: Command that was executed
        cwd: Working directory used
        start_time: Command start timestamp
        end_time: Command end timestamp
        exit_code: Command exit code
        stdout: Captured stdout
        stderr: Captured stderr
    """
    with open(log_path, "w", encoding="utf-8") as f:
        f.write("=" * 80 + "\n")
        f.write("COMMAND LOG\n")
        f.write("=" * 80 + "\n\n")
        
        f.write(f"Command:      {' '.join(command)}\n")
        f.write(f"Working Dir:  {cwd}\n")
        f.write(f"Start Time:   {start_time.strftime('%Y-%m-%d %H:%M:%S')}\n")
        f.write(f"End Time:     {end_time.strftime('%Y-%m-%d %H:%M:%S')}\n")
        f.write(f"Duration:     {(end_time - start_time).total_seconds():.2f} seconds\n")
        f.write(f"Exit Code:    {exit_code}\n")
        f.write("\n")
        
        f.write("=" * 80 + "\n")
        f.write("STDOUT\n")
        f.write("=" * 80 + "\n")
        if stdout:
            f.write(stdout)
            if not stdout.endswith("\n"):
                f.write("\n")
        else:
            f.write("(no output)\n")
        f.write("\n")
        
        f.write("=" * 80 + "\n")
        f.write("STDERR\n")
        f.write("=" * 80 + "\n")
        if stderr:
            f.write(stderr)
            if not stderr.endswith("\n"):
                f.write("\n")
        else:
            f.write("(no output)\n")
        f.write("\n")
        
        f.write("=" * 80 + "\n")
        f.write("END OF LOG\n")
        f.write("=" * 80 + "\n")


def main():
    parser = argparse.ArgumentParser(
        description="Run a command and capture its output to a timestamped log file.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s -- echo "hello world"
  %(prog)s --echo -- python script.py
  %(prog)s --cwd /path/to/dir -- npm install
  %(prog)s --timeout 60 -- long-running-command
  %(prog)s --echo --timeout 30 -- pytest

Note: The -- separator is required before the command.
        """,
    )
    
    parser.add_argument(
        "--echo",
        action="store_true",
        help="Also print stdout/stderr to console while capturing",
    )
    
    parser.add_argument(
        "--cwd",
        type=str,
        metavar="PATH",
        help="Run the command in a specific directory",
    )
    
    parser.add_argument(
        "--timeout",
        type=int,
        metavar="N",
        help="Optional timeout in seconds",
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Command Logger {get_version()}",
    )
    
    parser.add_argument(
        "command",
        nargs=argparse.REMAINDER,
        help="Command to run (after --)",
    )
    
    args = parser.parse_args()
    
    # Validate command
    if not args.command:
        print("Error: No command provided. Use -- before the command.", file=sys.stderr)
        print("\nExample: python kw_command_logger.py -- echo hello", file=sys.stderr)
        parser.print_help(sys.stderr)
        return 1
    
    # Remove the -- separator if present
    command = args.command
    if command and command[0] == "--":
        command = command[1:]
    
    if not command:
        print("Error: No command provided after --", file=sys.stderr)
        return 1
    
    # Validate timeout
    if args.timeout is not None and args.timeout < 1:
        print("Error: --timeout must be at least 1 second", file=sys.stderr)
        return 1
    
    # Determine working directory
    if args.cwd:
        cwd = Path(args.cwd).resolve()
        if not cwd.exists():
            print(f"Error: Directory does not exist: {cwd}", file=sys.stderr)
            return 1
        if not cwd.is_dir():
            print(f"Error: Path is not a directory: {cwd}", file=sys.stderr)
            return 1
    else:
        cwd = Path.cwd()
    
    # Generate log filename
    log_filename = get_log_filename()
    log_path = cwd / log_filename
    
    # Run command
    start_time = datetime.now()
    exit_code, stdout, stderr = run_command(command, cwd, args.timeout, args.echo)
    end_time = datetime.now()
    
    # Write log file
    try:
        write_log_file(log_path, command, cwd, start_time, end_time, exit_code, stdout, stderr)
    except Exception as e:
        print(f"Error: Failed to write log file: {e}", file=sys.stderr)
        return 1
    
    # Print summary
    print(f"✓ Log written to: {log_path}")
    print(f"✓ Exit code: {exit_code}")
    
    # Exit with the same code as the command
    return exit_code


if __name__ == "__main__":
    sys.exit(main())
