#!/usr/bin/env python3
"""
KW Env File Generator
Generate a .env file from a .env.example template.

Part of the Kindware toolset.
"""

import argparse
import secrets
import string
import sys
from pathlib import Path
from typing import List, Tuple


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


def generate_secret(length: int = 32) -> str:
    """
    Generate a URL-safe random string.
    
    Args:
        length: Length of the secret to generate
        
    Returns:
        Random URL-safe string
    """
    alphabet = string.ascii_letters + string.digits + "-_"
    return "".join(secrets.choice(alphabet) for _ in range(length))


def should_generate_secret(value: str) -> bool:
    """
    Check if a value should be replaced with a generated secret.
    
    Args:
        value: The value to check
        
    Returns:
        True if value contains generation placeholders
    """
    value_upper = value.upper()
    return "<GENERATE>" in value_upper or "<RANDOM>" in value_upper


def process_line(line: str, secret_length: int, verbose: bool) -> Tuple[str, bool]:
    """
    Process a single line from .env.example.
    
    Args:
        line: Line to process
        secret_length: Length of secrets to generate
        verbose: Whether to print verbose output
        
    Returns:
        Tuple of (processed_line, was_generated)
    """
    # Preserve comments and blank lines as-is
    if line.strip().startswith("#") or not line.strip():
        return line, False
    
    # Check if line contains an assignment
    if "=" not in line:
        return line, False
    
    # Split on first = only
    key, value = line.split("=", 1)
    key = key.strip()
    value = value.strip()
    
    # Check if we should generate a secret
    if should_generate_secret(value):
        generated_value = generate_secret(secret_length)
        if verbose:
            print(f"  Generated secret for: {key}", file=sys.stderr)
        return f"{key}={generated_value}\n", True
    
    # Preserve the line as-is (including empty values)
    return line, False


def parse_env_example(env_example_path: Path, secret_length: int, verbose: bool) -> Tuple[List[str], int]:
    """
    Parse .env.example and generate .env content.
    
    Args:
        env_example_path: Path to .env.example file
        secret_length: Length of secrets to generate
        verbose: Whether to print verbose output
        
    Returns:
        Tuple of (list of lines, count of generated secrets)
    """
    try:
        with open(env_example_path, "r", encoding="utf-8") as f:
            lines = f.readlines()
    except Exception as e:
        raise RuntimeError(f"Failed to read .env.example: {e}")
    
    processed_lines = []
    generated_count = 0
    
    for line in lines:
        processed_line, was_generated = process_line(line, secret_length, verbose)
        processed_lines.append(processed_line)
        if was_generated:
            generated_count += 1
    
    return processed_lines, generated_count


def main():
    parser = argparse.ArgumentParser(
        description="Generate a .env file from a .env.example template.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s /path/to/project
  %(prog)s . --verbose
  %(prog)s /path/to/project --force
  %(prog)s . --dry-run
  %(prog)s /path/to/project --length 64
        """,
    )
    
    parser.add_argument(
        "path",
        type=str,
        help="Directory containing .env.example",
    )
    
    parser.add_argument(
        "--force",
        action="store_true",
        help="Overwrite existing .env file",
    )
    
    parser.add_argument(
        "--length",
        type=int,
        default=32,
        metavar="N",
        help="Length of generated secrets (default: 32)",
    )
    
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Show what would be written without creating .env",
    )
    
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print detailed processing information",
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW Env Generator {get_version()}",
    )
    
    args = parser.parse_args()
    
    # Validate length
    if args.length < 1:
        print("Error: --length must be at least 1", file=sys.stderr)
        return 1
    
    # Validate path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        return 1
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        return 1
    
    # Check for .env.example
    env_example_path = target_path / ".env.example"
    if not env_example_path.exists():
        print(f"Error: .env.example not found in: {target_path}", file=sys.stderr)
        return 1
    
    # Check for existing .env
    env_path = target_path / ".env"
    if env_path.exists() and not args.force and not args.dry_run:
        print(f"Error: .env already exists at {env_path}", file=sys.stderr)
        print("Use --force to overwrite the existing file.", file=sys.stderr)
        return 1
    
    # Parse .env.example
    if args.verbose:
        print(f"Reading .env.example from: {env_example_path}", file=sys.stderr)
    
    try:
        processed_lines, generated_count = parse_env_example(
            env_example_path,
            args.length,
            args.verbose
        )
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        return 1
    
    # Dry-run mode: print what would be written
    if args.dry_run:
        print("[DRY-RUN] Would write the following to .env:")
        print()
        for line in processed_lines:
            print(line, end="")
        print()
        print(f"[DRY-RUN] Generated {generated_count} secret(s)")
        print(f"[DRY-RUN] Would write to: {env_path}")
        return 0
    
    # Write .env file
    try:
        with open(env_path, "w", encoding="utf-8") as f:
            f.writelines(processed_lines)
        
        if args.verbose:
            print(f"\n✓ Generated {generated_count} secret(s)", file=sys.stderr)
            print(f"✓ Wrote {len(processed_lines)} line(s)", file=sys.stderr)
        
        print(f"✓ Created .env at: {env_path}")
        return 0
    
    except Exception as e:
        print(f"Error: Failed to write .env: {e}", file=sys.stderr)
        return 1


if __name__ == "__main__":
    sys.exit(main())
