#!/usr/bin/env python3
"""
KW Git Ignore Builder
Generate a .gitignore file based on selected project types.

Part of the Kindware toolset.
"""

import argparse
import sys
from pathlib import Path
from typing import List


def get_version() -> str:
    """
    Read version from VERSION file in the same directory as this script.
    
    Returns:
        Version string (defaults to "0.0.0" if file is missing or unreadable)
    """
    try:
        version_file = Path(__file__).parent / "VERSION"
        return version_file.read_text().strip()
    except (FileNotFoundError, PermissionError, OSError):
        return "0.0.0"


# Template definitions
TEMPLATES = {
    "os_junk": [
        "# OS junk",
        ".DS_Store",
        "Thumbs.db",
        "Desktop.ini",
    ],
    "python": [
        "# Python",
        "__pycache__/",
        "*.py[cod]",
        "*.pyo",
        ".pytest_cache/",
        ".mypy_cache/",
        ".venv/",
        "venv/",
    ],
    "node": [
        "# Node",
        "node_modules/",
        "npm-debug.log*",
        "yarn-debug.log*",
        "yarn-error.log*",
        "dist/",
        "build/",
    ],
    "dotnet": [
        "# .NET",
        "bin/",
        "obj/",
        "*.user",
        "*.suo",
        ".vs/",
    ],
    "php": [
        "# PHP",
        "vendor/",
        ".phpunit.result.cache",
    ],
    "generic_logs_temp": [
        "# Generic logs and temp files",
        "*.log",
        "*.tmp",
        "*.swp",
    ],
}


def build_gitignore_content(include_blocks: List[str]) -> str:
    """
    Build the .gitignore file content from selected template blocks.
    
    Args:
        include_blocks: List of block names to include
        
    Returns:
        Complete .gitignore content as a string
    """
    # Always include OS junk and generic logs/temp
    blocks_to_include = ["os_junk"]
    
    # Add requested language blocks
    for block in include_blocks:
        if block in TEMPLATES:
            blocks_to_include.append(block)
    
    # Always add generic logs/temp at the end
    blocks_to_include.append("generic_logs_temp")
    
    # Remove duplicates while preserving order
    seen = set()
    unique_blocks = []
    for block in blocks_to_include:
        if block not in seen:
            seen.add(block)
            unique_blocks.append(block)
    
    # Build content
    content_lines = []
    for i, block_name in enumerate(unique_blocks):
        if i > 0:
            content_lines.append("")  # Blank line between blocks
        content_lines.extend(TEMPLATES[block_name])
    
    return "\n".join(content_lines) + "\n"


def main():
    parser = argparse.ArgumentParser(
        description="Generate a .gitignore file based on selected project types.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s /path/to/project --python
  %(prog)s /path/to/project --node --verbose
  %(prog)s /path/to/project --dotnet --force
  %(prog)s /path/to/project --all
  %(prog)s .
        """,
    )
    
    parser.add_argument(
        "path",
        type=str,
        help="Directory where .gitignore will be created",
    )
    
    parser.add_argument(
        "--python",
        action="store_true",
        help="Include Python template block",
    )
    
    parser.add_argument(
        "--node",
        action="store_true",
        help="Include Node template block",
    )
    
    parser.add_argument(
        "--dotnet",
        action="store_true",
        help="Include .NET template block",
    )
    
    parser.add_argument(
        "--php",
        action="store_true",
        help="Include PHP template block",
    )
    
    parser.add_argument(
        "--all",
        action="store_true",
        help="Include all language template blocks (Python, Node, .NET, PHP)",
    )
    
    parser.add_argument(
        "--force",
        action="store_true",
        help="Overwrite existing .gitignore file",
    )
    
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Print detailed information about what is being included",
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version=f"KW GitIgnore Builder {get_version()}",
    )
    
    args = parser.parse_args()
    
    # Validate path
    target_path = Path(args.path).resolve()
    
    if not target_path.exists():
        print(f"Error: Path does not exist: {target_path}", file=sys.stderr)
        return 1
    
    if not target_path.is_dir():
        print(f"Error: Path is not a directory: {target_path}", file=sys.stderr)
        return 1
    
    # Check for existing .gitignore
    gitignore_path = target_path / ".gitignore"
    if gitignore_path.exists() and not args.force:
        print(f"Error: .gitignore already exists at {gitignore_path}", file=sys.stderr)
        print("Use --force to overwrite the existing file.", file=sys.stderr)
        return 1
    
    # Determine which blocks to include
    include_blocks = []
    
    if args.all:
        include_blocks.extend(["python", "node", "dotnet", "php"])
    else:
        if args.python:
            include_blocks.append("python")
        if args.node:
            include_blocks.append("node")
        if args.dotnet:
            include_blocks.append("dotnet")
        if args.php:
            include_blocks.append("php")
    
    # Verbose output
    if args.verbose:
        print("Generating .gitignore with the following blocks:")
        print("  - OS junk (always included)")
        for block in include_blocks:
            print(f"  - {block.capitalize()}")
        print("  - Generic logs/temp (always included)")
        print()
    
    # Build and write .gitignore
    content = build_gitignore_content(include_blocks)
    
    try:
        gitignore_path.write_text(content, encoding="utf-8")
        print(f"✓ Created .gitignore at: {gitignore_path}")
        
        if args.verbose:
            print(f"\nGenerated {len(content.splitlines())} lines")
        
        return 0
    
    except Exception as e:
        print(f"Error: Failed to write .gitignore: {e}", file=sys.stderr)
        return 1


if __name__ == "__main__":
    sys.exit(main())
